/* Source Installer, Copyright (c) 2005 Claudio Fontana

   srcspawn.c - process spawning and logging

   This program is free software; you can redistribute it and/or modify
       it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
       (at your option) any later version.

   This program is distributed in the hope that it will be useful,
       but WITHOUT ANY WARRANTY; without even the implied warranty of
       MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
       GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
       along with this program (look for the file called COPYING);
       if not, write to the Free Software Foundation, Inc.,
           51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

       You can contact the author (Claudio Fontana) by sending a mail
       to claudio@gnu.org
*/

#include "src_stdinc.h"

struct srcinst_spawn_id {
  pid_t pid;			/* this is the process id that is running */
  char* argz;			/* argz of the running process */
  int in;			/* write descr to inject input to program(UNUSED) */
  int out;			/* read descr to capture program output */
  int err;			/* read descr to capture program errors */
  char freeme;			/* free on wait? */
};

int srcinst_get_spawn_in(SRCINST_SPAWN_ID id) {
  return ((struct srcinst_spawn_id*)id)->in;
}
int srcinst_get_spawn_out(SRCINST_SPAWN_ID id) {
  return ((struct srcinst_spawn_id*)id)->out;
}
int srcinst_get_spawn_err(SRCINST_SPAWN_ID id) {
  return ((struct srcinst_spawn_id*)id)->err;
}
pid_t srcinst_get_spawn_pid(SRCINST_SPAWN_ID id) {
  return ((struct srcinst_spawn_id*)id)->pid;
}

static void _process_callback_list(struct _callback_list* list, char* line) {
  struct _callback_node* this;

  for (this = list->first; this; this = this->next) {
    this->callback(line);
  }
}  

static void _capture_spawn_output(struct srcinst_spawn_id* id) {
  char buffer[SRCINST_BUFSIZE];
  struct _callback_list* callbacks[2];
  int* fd[2]; int max;
  FILE* files[2];
  fd_set toread;

  fd[0] = &(id->out); fd[1] = &(id->err);
  if (!(files[0] = fdopen(id->out, "r")))
    return;
  if (!(files[1] = fdopen(id->err, "r"))) {
    fclose(files[0]); id->out = -1;
    return;
  }

  callbacks[0] = &_srcinst_state.stdout_callbacks;
  callbacks[1] = &_srcinst_state.stderr_callbacks;

  while (1) {
    int i; struct timeval tv; int rv;

    FD_ZERO(&toread); max = -1;
    for (i = 0; i < 2; i++) {
      if (*(fd[i]) != -1) {
	FD_SET(*(fd[i]), &toread);
	if (*(fd[i]) > max)
	  max = *(fd[i]);
      }
    }

    tv.tv_sec = 0; tv.tv_usec = _srcinst_state.beat_delay;

    if ((rv = select(max + 1, &toread, 0, 0, &tv)) == -1) {
      for (i = 0; i < 2; i++) {
	if (files[i]) {
	  fclose(files[i]); files[i] = 0;
	  *(fd[i]) = -1;
	}
      }
      return;
    }

    if (rv == 0) {
      _process_callback_list(&_srcinst_state.beat_callbacks, 0);
    }

    for (i = 0; i < 2; i++) {
      if (*(fd[i]) != -1 && FD_ISSET(*(fd[i]), &toread)) {
	if (!srcinst_readline(buffer, SRCINST_BUFSIZE, files[i])) {
	  fclose(files[i]); files[i] = 0;
	  *(fd[i]) = -1;
	  /*
	    ssize_t size;
	    size = read(*(fd[i]), buffer, SRCINST_BUFSIZE - 1);
	    if (size <= 0) {
	    close(*(fd[i])); *(fd[i]) = -1;
	  */
	} else {
	  _process_callback_list(callbacks[i], buffer);
	}
      }
    }

    if (*(fd[0]) == -1 && *(fd[1]) == -1)
      return;
  }
}

static int _spawn_aux(struct srcinst_spawn_id* id, char* argz, char** args) {
  int pipe_out[2]; int pipe_err[2];
  int i; int rv = 0;
  char* line;

  pipe_out[0] = pipe_out[1] = pipe_err[0] = pipe_err[1] = -1;
  id->in = id->out = id->err = -1; id->pid = -1; 
  id->freeme = 0;
  id->argz = srcinst_strdup(argz);

  if (pipe(pipe_out) != 0 || pipe(pipe_err) != 0)
    goto end_proc;

  line = srcinst_strdup(args[0]);
  for (i = 1; args[i]; i++) {
    line = srcinst_append(line, " ");
    line = srcinst_append(line, args[i]);
  }

  if (_srcinst_state.log_spawn) {
    fprintf(stdout, "spawn: %s\n", line);
  }

  _process_callback_list(&_srcinst_state.spawn_callbacks, line);
  free(line);
    
  if ((id->pid = fork()) < 0)
    goto end_proc;

  else if (id->pid == 0) {
     close(1); close(2);
     dup(pipe_out[1]); close(pipe_out[1]);
     dup(pipe_err[1]); close(pipe_err[1]);
     execv(id->argz, args);
     exit(-1);
  }

  /* parent */
  rv = 1;

 end_proc:

  if (!rv) {
    srcinst_warning(SRCINST_ERR_CORE, "could not spawn", id->argz);

    if (pipe_out[0] != -1)
      { close(pipe_out[0]); pipe_out[0] = -1; }

    if (pipe_err[0] != -1)
      { close(pipe_err[0]); pipe_err[0] = -1; }

    if (id->argz)
      { free(id->argz); id->argz = 0; }
  }

  if (pipe_out[1] != -1)
    close(pipe_out[1]);
  
  if (pipe_err[1] != -1)
    close(pipe_err[1]);

  id->out = pipe_out[0]; id->err = pipe_err[0];
  return rv;
}

static int _spawn_aux_ap(struct srcinst_spawn_id* id, char* args, va_list ap) {
  int rv; int argc = 0; char** strings = 0; 

  while (1) {
    strings = srcinst_realloc(strings, (argc + 1) * sizeof(char*));
    strings[argc] = args;
    argc++;
    if (!args)
      break;
    args = va_arg(ap, char*);
  }

  rv = _spawn_aux(id, strings[0], strings);
  free(strings);
  return rv;
}

/* spawn a process with fullpath, and arbitrary n of arguments (use 0 as last),
   redirecting stdout and stderr. Returns 0 (error) or 1 (ok). The
   passed srcinst_spawn_id structure is overwritten with the process id and
   the redirected stdin, stdout */

SRCINST_SPAWN_ID srcinst_spawn(char* args, ...) {
  int rv; struct srcinst_spawn_id* id; va_list ap;
  va_start(ap, args);

  id = srcinst_malloc(sizeof(struct srcinst_spawn_id));
  if (!(rv = _spawn_aux_ap(id, args, ap))) {
    free(id); id = 0;
  }

  id->freeme = 1;
  va_end(ap);
  return id;
}

/* static data for callbacks */

static char* _callback_expect_pattern_str;
static SRCINST_REGEX _callback_expect_regex;
static SRCINST_MATCH _callback_expect_match;
static struct srcinst_string_list* _callback_string_list;

static void _callback_printout_with_argz(char* line) {
  struct srcinst_spawn_id* id; id = _srcinst_state.waiting_for;
  fprintf(stdout, "%s: %s\n", id->argz, line);
}

static void _callback_printerr_with_argz(char* line) {
  struct srcinst_spawn_id* id; id = _srcinst_state.waiting_for;
  fprintf(stderr, "%s: %s\n", id->argz, line);
}

/* wait for a process to end execution. This may seem blocking,
   but all the output callbacks will be called, and the heartbeat
   callbacks will be called too at each beat.

   Additionally, the process id is set in the global state (waiting_for),
   so the process can be controlled from the outside too. This is 
   not used in any of the sourceinstall programs.
 */

int srcinst_wait(SRCINST_SPAWN_ID sid) {
  int status;
  struct srcinst_spawn_id* id;

  _srcinst_state.waiting_for = id = sid; /* waiting for... */

  if (_srcinst_state.log_stdout) {
    srcinst_register_stdout_callback(&_callback_printout_with_argz);
  }

  if (_srcinst_state.log_stderr) {
    srcinst_register_stderr_callback(&_callback_printerr_with_argz);
  }
  _capture_spawn_output(id);

  if (_srcinst_state.log_stdout) {
    srcinst_unregister_stdout_callback(&_callback_printout_with_argz);
  }

  if (_srcinst_state.log_stderr) {
    srcinst_unregister_stderr_callback(&_callback_printerr_with_argz);
  }

  if (id->in != -1)
    { close(id->in); id->in = -1; }
  if (id->out != -1)
    { close(id->out); id->out = -1; }
  if (id->err != -1)
    { close(id->err); id->err = -1; }
  
  if (id->argz)
    { free(id->argz); id->argz = 0; }

  _srcinst_state.waiting_for = 0; /* ...end waiting. */
  
  if (waitpid(id->pid, &status, 0) > 0) {
    if (WIFEXITED(status)) {
      if (id->freeme) free(id);
      return WEXITSTATUS(status) == 0;
    }
  }
  if (id->freeme) {
    free(id); id->freeme = 0;
  }
  return 0;
}

int srcinst_spawn_wait(char* args, ...) {
  int rv; va_list ap;
  struct srcinst_spawn_id id;
  va_start(ap, args);

  if ((rv = _spawn_aux_ap(&id, args, ap)))
    rv = srcinst_wait(&id);
  
  va_end(ap);
  return rv;
}

int srcinst_spawn_waitve(char* argz, char** args) {
  int rv; struct srcinst_spawn_id id;

  if ((rv = _spawn_aux(&id, argz, args)))
    rv = srcinst_wait(&id);

  return rv;
}

static void _callback_expect_pattern(char* line) {
  if (!_callback_expect_pattern_str) {
    if (srcinst_exec_regex(_callback_expect_regex, line, _callback_expect_match)) {
      /* found. Alloc submatch in the heap. */
      _callback_expect_pattern_str = srcinst_get_match(_callback_expect_match, 1);
    }
  }
}

char* srcinst_spawn_wait_expect(char* pattern, char* args, ...) {
  int rv; va_list ap;
  struct srcinst_spawn_id id;
  va_start(ap, args);

  _callback_expect_pattern_str = 0;

  if (!(rv = _spawn_aux_ap(&id, args, ap)))
    goto end_proc;

  srcinst_register_stdout_callback(&_callback_expect_pattern);

  _callback_expect_regex = srcinst_init_regex(pattern, -1);
  _callback_expect_match = srcinst_init_match(2);

  if (!srcinst_wait(&id)) {
    if (_callback_expect_pattern_str) {
      free(_callback_expect_pattern_str);
      _callback_expect_pattern_str = 0;
    }
  }

  srcinst_unregister_stdout_callback(&_callback_expect_pattern);

  srcinst_free_regex(_callback_expect_regex);
  srcinst_free_match(_callback_expect_match);

 end_proc:
  va_end(ap);
  return _callback_expect_pattern_str;
}

static void _callback_expect_all(char* line) {

  if (_callback_expect_pattern_str) {
    _callback_expect_pattern_str = srcinst_append(_callback_expect_pattern_str,
						  "\n");
  }
  _callback_expect_pattern_str = srcinst_append(_callback_expect_pattern_str,
						line);
}

char* srcinst_spawn_wait_expect_all(char* args, ...) { 
  int rv; va_list ap;
  struct srcinst_spawn_id id;
  va_start(ap, args);

  _callback_expect_pattern_str = 0;

  if (!(rv = _spawn_aux_ap(&id, args, ap)))
    goto end_proc;

  srcinst_register_stdout_callback(&_callback_expect_all);

  if (!srcinst_wait(&id)) {
    if (_callback_expect_pattern_str) {
      free(_callback_expect_pattern_str);
      _callback_expect_pattern_str = 0;
    }
  }

  srcinst_unregister_stdout_callback(&_callback_expect_all);
  
 end_proc:

  va_end(ap);
  return _callback_expect_pattern_str;
}

static void _callback_expect_lines(char* line) {
  srcinst_add_string_list(_callback_string_list, line);
}

int srcinst_spawn_wait_expect_lines(struct srcinst_string_list* list, char* args, ...) { 
  int rv; va_list ap;
  struct srcinst_spawn_id id;
  va_start(ap, args);

  rv = 0;
  srcinst_init_string_list(list);
  _callback_string_list = list;
  
  if (!(rv = _spawn_aux_ap(&id, args, ap)))
    goto end_proc;

  srcinst_register_stdout_callback(&_callback_expect_lines);

  if (!srcinst_wait(&id)) {
    srcinst_free_string_list(list);    
  }

  srcinst_unregister_stdout_callback(&_callback_expect_lines);

 end_proc:
  va_end(ap);
  return rv;
}

static void _callback_expect_strip(char* line) {
  int offset;

  if (srcinst_exec_regex(_callback_expect_regex, line, _callback_expect_match) && 
      (offset = srcinst_get_match_offset(_callback_expect_match, 1)) != -1) {
    /* found. Strip, then add to string list */
    char* dest, *src;
    dest = line + offset;
    src = dest + srcinst_get_match_len(_callback_expect_match, 1);
    memmove(dest, src, strlen(dest) + 1);
    srcinst_add_string_list(_callback_string_list, line);
  }
}

int srcinst_spawn_wait_expect_strip(char* pattern, struct srcinst_string_list* list, char* args, ...) {
  int rv; va_list ap;
  struct srcinst_spawn_id id;
  va_start(ap, args);

  rv = 0;
  srcinst_init_string_list(list);
  _callback_string_list = list;

  if (!(rv = _spawn_aux_ap(&id, args, ap)))
    goto end_proc;

  srcinst_register_stdout_callback(&_callback_expect_strip);

  _callback_expect_regex = srcinst_init_regex(pattern, -1);
  _callback_expect_match = srcinst_init_match(2);

  if (!(rv = srcinst_wait(&id))) {
    srcinst_free_string_list(list);
  }

  srcinst_unregister_stdout_callback(&_callback_expect_strip);

  srcinst_free_regex(_callback_expect_regex);
  srcinst_free_match(_callback_expect_match);

 end_proc:
  va_end(ap);
  return rv;
}
