/* Source Installer, Copyright (c) 2005 Claudio Fontana

   string_list.c - simple string set

   This program is free software; you can redistribute it and/or modify
       it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
       (at your option) any later version.

   This program is distributed in the hope that it will be useful,
       but WITHOUT ANY WARRANTY; without even the implied warranty of
       MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
       GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
       along with this program (look for the file called COPYING);
       if not, write to the Free Software Foundation, Inc.,
           51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

       You can contact the author (Claudio Fontana) by sending a mail
       to claudio@gnu.org
*/

#include "src_stdinc.h"

void srcinst_init_string_list(struct srcinst_string_list* l) {
  l->count = 0; l->strings = 0;
}

void srcinst_free_string_list(struct srcinst_string_list* l) {
  int i;

  for (i = 0; i < l->count; i++) {
    if (l->strings[i])
      free(l->strings[i]);
  }

  if (l->strings) {
    free(l->strings);
  }

  l->count = 0; l->strings = 0;
}

int srcinst_glob_string_list(struct srcinst_string_list* l, char* dirname, char* pattern) {
  DIR* dir; struct dirent* entry;
  
  l->count = 0; l->strings = 0;

  if (!(dir = opendir(dirname))) {
    return 0;
  }

  while ((entry = readdir(dir))) {
    if (strcmp(entry->d_name, ".") != 0 && strcmp(entry->d_name, "..") != 0)
      if (!pattern || fnmatch(pattern, entry->d_name, 0) == 0)
	srcinst_add_string_list(l, entry->d_name);
  }
  
  if (closedir(dir) != 0) {
    return 0;
  }

  return 1;
}

void srcinst_add_string_list(struct srcinst_string_list* l, char* string) {
  l->count++;
  l->strings = srcinst_realloc(l->strings, l->count * sizeof(char*));
  l->strings[l->count - 1] = string ? srcinst_strdup(string) : 0;
}

/* deep copy string list */

void srcinst_copy_string_list(struct srcinst_string_list* dest,
			      struct srcinst_string_list* src) {
  int i;
  dest->count = src->count;
  dest->strings = srcinst_malloc(sizeof(char*) * src->count);
  
  for (i = 0; i < src->count; i++) {
    dest->strings[i] = srcinst_strdup(src->strings[i]);
  }
}

/* copy only the pointers. The char** is heap-allocated though. */

void srcinst_mirror_string_list(struct srcinst_string_list* dest,
				struct srcinst_string_list* src) {
  int i;
  dest->count = src->count;
  dest->strings = srcinst_malloc(sizeof(char*) * src->count);

  for (i = 0; i < src->count; i++) {
    dest->strings[i] = src->strings[i];
  }
}


/* purge list from "" files, and optionally from nonexistant files */

void srcinst_purge_string_list(struct srcinst_string_list* l, 
			       int purge_nonexistant) {
  int i, j; size_t n;
  char** new_strings;

  n = l->count;

  for (i = 0; i < l->count; i++) {
    char* filename = l->strings[i];

    if (*filename == '\0') {
      n--;

    } else if (purge_nonexistant && srcinst_file_ltype(filename) ==
	       SRCINST_TYPE_NONE) {
      /* mark list element for removal */
      *filename = '\0';
      n--;
    }
  }

  new_strings = srcinst_malloc(sizeof(char*) * n);
  
  for (i = 0, j = 0; i < l->count; i++) {
    char* filename = l->strings[i];
    if (*filename != '\0') {
      new_strings[j++] = filename;
    } else {
      free(filename);
    }
  }

  free(l->strings);
  l->strings = new_strings;
  l->count = n;
}

char* srcinst_collapse_string_list(struct srcinst_string_list* l, char* sep) {
  char* str; size_t* lens; size_t len_sep; int i;
  
  if (!l->count)
    return srcinst_strdup("");

  len_sep = strlen(sep);
  lens = srcinst_malloc(sizeof(size_t) * l->count);
  lens[0] = strlen(l->strings[0]);

  for (i = 1; i < l->count; i++) {
    lens[i] = lens[i - 1] + strlen(l->strings[i]) + len_sep;
  }

  str = srcinst_malloc(lens[l->count - 1] + 1);
  strcpy(str, l->strings[0]);

  for (i = 1; i < l->count; i++) {
    sprintf(str + lens[i - 1], "%s%s", sep, l->strings[i]);
  }

  free(lens);
  return str;
}

void srcinst_sort_string_list(struct srcinst_string_list* list,
			      int (*compare)(char**, char**)) {

  srcinst_qsort(list->strings, list->count, compare);
}

/* 

int srcinst_search_string_list(struct srcinst_string_list* list, char* str) {
  int i;
  
  for (i = 0; i < list->count; i++) {
    if (strcmp(str, list->strings[i]) == 0)
      return i;
  }
  
  return -1;
}
			       
*/

void srcinst_prefix_string_list(struct srcinst_string_list* list,
				char* prefix) {
  int i;
  
  for (i = 0; i < list->count; i++) {
    char *oldstring = list->strings[i];
    list->strings[i] = srcinst_strjoin(prefix, oldstring, 0);
    free(oldstring);
  }
}
